
#include <math.h>
#include <limits.h>
#include "csparse.h"


/********************************************************************************
 *                                                                              *
 *                            FUNCTION DEFINITIONS                              *
 *                                                                              *
 ********************************************************************************/

void *cs_malloc(int n, size_t size) {

	return (malloc(CS_MAX (n,1) * size));
}

void *cs_calloc(int n, size_t size) {

	return (calloc(CS_MAX (n,1), size));
}

void *cs_free(void *p) {
	if (p)
		free(p); /* free p if it is not already NULL */
	return (NULL); /* return NULL to simplify the use of cs_free */
}

void *cs_realloc(void *p, int n, size_t size, int *ok) {

	void *pnew;
	pnew = realloc(p, CS_MAX (n,1) * size); /* realloc the block */
	*ok = (pnew != NULL); /* realloc fails if pnew is NULL */
	return ((*ok) ? pnew : p); /* return original p if failure */
}

cs *cs_spfree(cs *A) {

	if (!A)
		return (NULL); /* do nothing if A already NULL */
	cs_free(A->p);
	cs_free(A->i);
	cs_free(A->x);
	return (cs *) (cs_free(A)); /* free the cs struct and return NULL */
}

csn *cs_nfree(csn *N) {

	if (!N)
		return (NULL); /* do nothing if N already NULL */
	cs_spfree(N->L);
	cs_spfree(N->U);
	cs_free(N->pinv);
	cs_free(N->B);
	return (csn *) (cs_free(N)); /* free the csn struct and return NULL */
}

css *cs_sfree(css *S) {

	if (!S)
		return (NULL); /* do nothing if S already NULL */
	cs_free(S->pinv);
	cs_free(S->q);
	cs_free(S->parent);
	cs_free(S->cp);
	cs_free(S->leftmost);
	return (css *) (cs_free(S)); /* free the css struct and return NULL */
}

cs *cs_done(cs *C, void *w, void *x, int ok) {

	cs_free(w); /* free workspace */
	cs_free(x);
	return (ok ? C : cs_spfree(C)); /* return result if OK, else free it */
}

int *cs_idone(int *p, cs *C, void *w, int ok) {

	cs_spfree(C); /* free temporary matrix */
	cs_free(w); /* free workspace */
	return (int *) (ok ? p : cs_free(p)); /* return result if OK, else free it */
}

csn *cs_ndone(csn *N, cs *C, void *w, void *x, int ok) {

	cs_spfree(C); /* free temporary matrix */
	cs_free(w); /* free workspace */
	cs_free(x);
	return (ok ? N : cs_nfree(N)); /* return result if OK, else free it */
}

cs *cs_spalloc(int m, int n, int nzmax, int values, int triplet) {

	cs *A = (cs *) cs_calloc(1, sizeof(cs)); /* allocate the cs struct */
	if (!A)
		return (NULL); /* out of memory */
	A->m = m; /* define dimensions and nzmax */
	A->n = n;
	A->nzmax = nzmax = CS_MAX (nzmax, 1);
	A->nz = triplet ? 0 : -1; /* allocate triplet or comp.col */
	A->p = (int *) cs_malloc(triplet ? nzmax : n + 1, sizeof(int));
	A->i = (int *) cs_malloc(nzmax, sizeof(int));
	A->x = (double *) (values ? cs_malloc(nzmax, sizeof(double)) : NULL);
	return ((!A->p || !A->i || (values && !A->x)) ? cs_spfree(A) : A);
}

int cs_sprealloc(cs *A, int nzmax) {

	int ok, oki, okj = 1, okx = 1;
	if (!A)
		return (0);
	if (nzmax <= 0)
		nzmax = (CS_CSC (A)) ? (A->p[A->n]) : A->nz;
	A->i = (int *) cs_realloc(A->i, nzmax, sizeof(int), &oki);
	if (CS_TRIPLET (A))
		A->p = (int *) cs_realloc(A->p, nzmax, sizeof(int), &okj);
	if (A->x)
		A->x = (double *) cs_realloc(A->x, nzmax, sizeof(double), &okx);
	ok = (oki && okj && okx);
	if (ok)
		A->nzmax = nzmax;
	return (ok);
}


cs *cs_compress(const cs *T) {

	int m, n, nz, p, k, *Cp, *Ci, *w, *Ti, *Tj;
	double *Cx, *Tx;
	cs *C;
	if (!CS_TRIPLET (T))
		return (NULL); /* check inputs */
	m = T->m;
	n = T->n;
	Ti = T->i;
	Tj = T->p;
	Tx = T->x;
	nz = T->nz;
	C = cs_spalloc(m, n, nz, Tx != NULL, 0); /* allocate result */
	w = (int *) cs_calloc(n, sizeof(int)); /* get workspace */
	if (!C || !w)
		return (cs_done(C, w, NULL, 0)); /* out of memory */
	Cp = C->p;
	Ci = C->i;
	Cx = C->x;
	for (k = 0; k < nz; k++)
		w[Tj[k]]++; /* column counts */
	cs_cumsum(Cp, w, n); /* column pointers */
	for (k = 0; k < nz; k++) {
		Ci[p = w[Tj[k]]++] = Ti[k]; /* A(i,j) is the pth entry in C */
		if (Cx)
			Cx[p] = Tx[k];
	}
	return (cs_done(C, w, NULL, 1)); /* success; free w and return C */
}

double cs_cumsum(int *p, int *c, int n) {

	int i, nz = 0;
	double nz2 = 0;
	if (!p || !c)
		return (-1); /* check inputs */
	for (i = 0; i < n; i++) {
		p[i] = nz;
		nz += c[i];
		nz2 += c[i]; /* also in double to avoid int overflow */
		c[i] = p[i]; /* also copy p[0..n-1] back into c[0..n-1]*/
	}
	p[n] = nz;
	return (nz2); /* return sum (c [0..n-1]) */
}

cs *cs_transpose(const cs *A, int values) {

	int p, q, j, *Cp, *Ci, n, m, *Ap, *Ai, *w;
	double *Cx, *Ax;
	cs *C;
	if (!CS_CSC (A))
		return (NULL); /* check inputs */
	m = A->m;
	n = A->n;
	Ap = A->p;
	Ai = A->i;
	Ax = A->x;
	C = cs_spalloc(n, m, Ap[n], values && Ax, 0); /* allocate result */
	w = (int *) cs_calloc(m, sizeof(int)); /* get workspace */
	if (!C || !w)
		return (cs_done(C, w, NULL, 0)); /* out of memory */
	Cp = C->p;
	Ci = C->i;
	Cx = C->x;
	for (p = 0; p < Ap[n]; p++)
		w[Ai[p]]++; /* row counts */
	cs_cumsum(Cp, w, m); /* row pointers */
	for (j = 0; j < n; j++) {
		for (p = Ap[j]; p < Ap[j + 1]; p++) {
			Ci[q = w[Ai[p]]++] = j; /* place A(i,j) as entry C(j,i) */
			if (Cx)
				Cx[q] = Ax[p];
		}
	}
	return (cs_done(C, w, NULL, 1)); /* success; free w and return C */
}

int cs_dupl(cs *A) {

	int i, j, p, q, nz = 0, n, m, *Ap, *Ai, *w;
	double *Ax;
	if (!CS_CSC (A))
		return (0); /* check inputs */
	m = A->m;
	n = A->n;
	Ap = A->p;
	Ai = A->i;
	Ax = A->x;
	w = (int *) cs_malloc(m, sizeof(int)); /* get workspace */
	if (!w)
		return (0); /* out of memory */
	for (i = 0; i < m; i++)
		w[i] = -1; /* row i not yet seen */
	for (j = 0; j < n; j++) {
		q = nz; /* column j will start at q */
		for (p = Ap[j]; p < Ap[j + 1]; p++) {
			i = Ai[p]; /* A(i,j) is nonzero */
			if (w[i] >= q) {
				Ax[w[i]] += Ax[p]; /* A(i,j) is a duplicate */
			} else {
				w[i] = nz; /* record where row i occurs */
				Ai[nz] = i; /* keep A(i,j) */
				Ax[nz++] = Ax[p];
			}
		}
		Ap[j] = q; /* record start of column j */
	}
	Ap[n] = nz; /* finalize A */
	cs_free(w); /* free workspace */
	return (cs_sprealloc(A, 0)); /* remove extra space from A */
}

int cs_pvec(const int *p, const double *b, double *x, int n) {

	int k;
	if (!x || !b)
		return (0); /* check inputs */
	for (k = 0; k < n; k++)
		x[k] = b[p ? p[k] : k];
	return (1);
}

int cs_ipvec(const int *p, const double *b, double *x, int n) {

	int k;
	if (!x || !b)
		return (0); /* check inputs */
	for (k = 0; k < n; k++)
		x[p ? p[k] : k] = b[k];
	return (1);
}

int *cs_pinv(int const *p, int n) {

	int k, *pinv;
	if (!p)
		return (NULL); /* p = NULL denotes identity */
	pinv = (int *) cs_malloc(n, sizeof(int)); /* allocate result */
	if (!pinv)
		return (NULL); /* out of memory */
	for (k = 0; k < n; k++)
		pinv[p[k]] = k;/* invert the permutation */
	return (pinv); /* return result */
}

cs *cs_symperm(const cs *A, const int *pinv, int values) {

	int i, j, p, q, i2, j2, n, *Ap, *Ai, *Cp, *Ci, *w;
	double *Cx, *Ax;
	cs *C;
	if (!CS_CSC (A))
		return (NULL); /* check inputs */
	n = A->n;
	Ap = A->p;
	Ai = A->i;
	Ax = A->x;
	C = cs_spalloc(n, n, Ap[n], values && (Ax != NULL), 0); /* alloc result*/
	w = (int *) cs_calloc(n, sizeof(int)); /* get workspace */
	if (!C || !w)
		return (cs_done(C, w, NULL, 0)); /* out of memory */
	Cp = C->p;
	Ci = C->i;
	Cx = C->x;
	for (j = 0; j < n; j++) /* count entries in each column of C */
	{
		j2 = pinv ? pinv[j] : j; /* column j of A is column j2 of C */
		for (p = Ap[j]; p < Ap[j + 1]; p++) {
			i = Ai[p];
			if (i > j)
				continue; /* skip lower triangular part of A */
			i2 = pinv ? pinv[i] : i; /* row i of A is row i2 of C */
			w[CS_MAX (i2, j2)]++; /* column count of C */
		}
	}
	cs_cumsum(Cp, w, n); /* compute column pointers of C */
	for (j = 0; j < n; j++) {
		j2 = pinv ? pinv[j] : j; /* column j of A is column j2 of C */
		for (p = Ap[j]; p < Ap[j + 1]; p++) {
			i = Ai[p];
			if (i > j)
				continue; /* skip lower triangular part of A*/
			i2 = pinv ? pinv[i] : i; /* row i of A is row i2 of C */
			Ci[q = w[CS_MAX (i2, j2)]++] = CS_MIN (i2, j2);
			if (Cx)
				Cx[q] = Ax[p];
		}
	}
	return (cs_done(C, w, NULL, 1)); /* success; free workspace, return C */
}

int cs_scatter(const cs *A, int j, double beta, int *w, double *x, int mark, cs *C, int nz) {

	int i, p, *Ap, *Ai, *Ci;
	double *Ax;
	if (!CS_CSC (A) || !w || !CS_CSC (C))
		return (-1); /* check inputs */
	Ap = A->p;
	Ai = A->i;
	Ax = A->x;
	Ci = C->i;
	for (p = Ap[j]; p < Ap[j + 1]; p++) {
		i = Ai[p]; /* A(i,j) is nonzero */
		if (w[i] < mark) {
			w[i] = mark; /* i is new entry in column j */
			Ci[nz++] = i; /* add i to pattern of C(:,j) */
			if (x)
				x[i] = beta * Ax[p]; /* x(i) = beta*A(i,j) */
		} else if (x)
			x[i] += beta * Ax[p]; /* i exists in C(:,j) already */
	}
	return (nz);
}

cs *cs_permute (const cs *A, const int *pinv, const int *q, int values)
{
    int t, j, k, nz = 0, m, n, *Ap, *Ai, *Cp, *Ci ;
    double *Cx, *Ax ;
    cs *C ;
    if (!CS_CSC (A)) return (NULL) ;    /* check inputs */
    m = A->m ; n = A->n ; Ap = A->p ; Ai = A->i ; Ax = A->x ;
    C = cs_spalloc (m, n, Ap [n], values && Ax != NULL, 0) ;  /* alloc result */
    if (!C) return (cs_done (C, NULL, NULL, 0)) ;   /* out of memory */
    Cp = C->p ; Ci = C->i ; Cx = C->x ;
    for (k = 0 ; k < n ; k++)
    {
        Cp [k] = nz ;                   /* column k of C is column q[k] of A */
        j = q ? (q [k]) : k ;
        for (t = Ap [j] ; t < Ap [j+1] ; t++)
        {
            if (Cx) Cx [nz] = Ax [t] ;  /* row i of A is row pinv[i] of C */
            Ci [nz++] = pinv ? (pinv [Ai [t]]) : Ai [t] ;
        }
    }
    Cp [n] = nz ;                       /* finalize the last column of C */
    return (cs_done (C, NULL, NULL, 1)) ;
}

cs *cs_add(const cs *A, const cs *B, double alpha, double beta) {

	int p, j, nz = 0, anz, *Cp, *Ci, *Bp, m, n, bnz, *w, values;
	double *x, *Bx, *Cx;
	cs *C;
	if (!CS_CSC (A) || !CS_CSC (B))
		return (NULL); /* check inputs */
	if (A->m != B->m || A->n != B->n)
		return (NULL);
	m = A->m;
	anz = A->p[A->n];
	n = B->n;
	Bp = B->p;
	Bx = B->x;
	bnz = Bp[n];
	w = (int *) cs_calloc(m, sizeof(int)); /* get workspace */
	values = (A->x != NULL) && (Bx != NULL);
	x = (double *) (values ? cs_malloc(m, sizeof(double)) : NULL); /* get workspace */
	C = cs_spalloc(m, n, anz + bnz, values, 0); /* allocate result*/
	if (!C || !w || (values && !x))
		return (cs_done(C, w, x, 0));
	Cp = C->p;
	Ci = C->i;
	Cx = C->x;
	for (j = 0; j < n; j++) {
		Cp[j] = nz; /* column j of C starts here */
		nz = cs_scatter(A, j, alpha, w, x, j + 1, C, nz); /* alpha*A(:,j)*/
		nz = cs_scatter(B, j, beta, w, x, j + 1, C, nz); /* beta*B(:,j) */
		if (values)
			for (p = Cp[j]; p < nz; p++)
				Cx[p] = x[Ci[p]];
	}
	Cp[n] = nz; /* finalize the last column of C */
	cs_sprealloc(C, 0); /* remove extra space from C */
	return (cs_done(C, w, x, 1)); /* success; free workspace, return C */
}

cs *cs_multiply(const cs *A, const cs *B) {

	int p, j, nz = 0, anz, *Cp, *Ci, *Bp, m, n, bnz, *w, values, *Bi;
	double *x, *Bx, *Cx;
	cs *C;
	if (!CS_CSC (A) || !CS_CSC (B))
		return (NULL); /* check inputs */
	if (A->n != B->m)
		return (NULL);
	m = A->m;
	anz = A->p[A->n];
	n = B->n;
	Bp = B->p;
	Bi = B->i;
	Bx = B->x;
	bnz = Bp[n];
	w = (int *) cs_calloc(m, sizeof(int)); /* get workspace */
	values = (A->x != NULL) && (Bx != NULL);
	x = (double *) (values ? cs_malloc(m, sizeof(double)) : NULL); /* get workspace */
	C = cs_spalloc(m, n, anz + bnz, values, 0); /* allocate result */
	if (!C || !w || (values && !x))
		return (cs_done(C, w, x, 0));
	Cp = C->p;
	for (j = 0; j < n; j++) {
		if (nz + m > C->nzmax && !cs_sprealloc(C, 2 * (C->nzmax) + m)) {
			return (cs_done(C, w, x, 0)); /* out of memory */
		}
		Ci = C->i;
		Cx = C->x; /* C->i and C->x may be reallocated */
		Cp[j] = nz; /* column j of C starts here */
		for (p = Bp[j]; p < Bp[j + 1]; p++) {
			nz = cs_scatter(A, Bi[p], Bx ? Bx[p] : 1, w, x, j + 1, C, nz);
		}
		if (values)
			for (p = Cp[j]; p < nz; p++)
				Cx[p] = x[Ci[p]];
	}
	Cp[n] = nz; /* finalize the last column of C */
	cs_sprealloc(C, 0); /* remove extra space from C */
	return (cs_done(C, w, x, 1)); /* success; free workspace, return C */
}

int cs_gaxpy (const cs *A, const double *x, double *y)
{
    int p, j, n, *Ap, *Ai ;
    double *Ax ;
    if (!CS_CSC (A) || !x || !y) return (0) ;       /* check inputs */
    n = A->n ; Ap = A->p ; Ai = A->i ; Ax = A->x ;
    for (j = 0 ; j < n ; j++)
    {
        for (p = Ap [j] ; p < Ap [j+1] ; p++)
        {
            y [Ai [p]] += Ax [p] * x [j] ;
        }
    }
    return (1) ;
}

double cs_norm (const cs *A)
{
    int p, j, n, *Ap ;
    double *Ax,  norm = 0, s ;
    if (!CS_CSC (A) || !A->x) return (-1) ;             /* check inputs */
    n = A->n ; Ap = A->p ; Ax = A->x ;
    for (j = 0 ; j < n ; j++)
    {
        for (s = 0, p = Ap [j] ; p < Ap [j+1] ; p++) s += fabs (Ax [p]) ;
        norm = CS_MAX (norm, s) ;
    }
    return (norm) ;
}

int cs_fkeep(cs *A, int(*fkeep)(int, int, double, void *), void *other) {

	int j, p, nz = 0, n, *Ap, *Ai;
	double *Ax;
	if (!CS_CSC (A) || !fkeep)
		return (-1); /* check inputs */
	n = A->n;
	Ap = A->p;
	Ai = A->i;
	Ax = A->x;
	for (j = 0; j < n; j++) {
		p = Ap[j]; /* get current location of col j */
		Ap[j] = nz; /* record new location of col j */
		for (; p < Ap[j + 1]; p++) {
			if (fkeep(Ai[p], j, Ax ? Ax[p] : 1, other)) {
				if (Ax)
					Ax[nz] = Ax[p]; /* keep A(i,j) */
				Ai[nz++] = Ai[p];
			}
		}
	}
	Ap[n] = nz; /* finalize A */
	cs_sprealloc(A, 0); /* remove extra space from A */
	return (nz);
}

csn *cs_lu (const cs *A, const css *S, double tol)
{
    cs *L, *U ;
    csn *N ;
    double pivot, *Lx, *Ux, *x,  a, t ;
    int *Lp, *Li, *Up, *Ui, *pinv, *xi, *q, n, ipiv, k, top, p, i, col, lnz,unz;
    if (!CS_CSC (A) || !S) return (NULL) ;          /* check inputs */
    n = A->n ;
    q = S->q ; lnz = S->lnz ; unz = S->unz ;
    x = cs_malloc (n, sizeof (double)) ;            /* get double workspace */
    xi = cs_malloc (2*n, sizeof (int)) ;            /* get csi workspace */
    N = cs_calloc (1, sizeof (csn)) ;               /* allocate result */

    //if ( !x || !xi || !N ) printf ( "x/xi/N not allocated\n" );
    //fflush ( stdout );

    if (!x || !xi || !N) return (cs_ndone (N, NULL, xi, x, 0)) ;
    N->L = L = cs_spalloc (n, n, lnz, 1, 0) ;       /* allocate result L */
    N->U = U = cs_spalloc (n, n, unz, 1, 0) ;       /* allocate result U */
    N->pinv = pinv = cs_malloc (n, sizeof (int)) ;  /* allocate result pinv */

   // if ( !L || !U || !pinv ) printf ( "L/U/pinv not allocated\n" );
   // fflush ( stdout );

    if (!L || !U || !pinv) return (cs_ndone (N, NULL, xi, x, 0)) ;
    Lp = L->p ; Up = U->p ;
    for (i = 0 ; i < n ; i++) x [i] = 0 ;           /* clear workspace */
    for (i = 0 ; i < n ; i++) pinv [i] = -1 ;       /* no rows pivotal yet */
    for (k = 0 ; k <= n ; k++) Lp [k] = 0 ;         /* no cols of L yet */
    lnz = unz = 0 ;
    for (k = 0 ; k < n ; k++)       /* compute L(:,k) and U(:,k) */
    {
        /* --- Triangular solve --------------------------------------------- */
        Lp [k] = lnz ;              /* L(:,k) starts here */
        Up [k] = unz ;              /* U(:,k) starts here */
        if ((lnz + n > L->nzmax && !cs_sprealloc (L, 2*L->nzmax + n)) ||
            (unz + n > U->nzmax && !cs_sprealloc (U, 2*U->nzmax + n)))
        {
        	//printf ( "Triangular Fail\n" );
            return (cs_ndone (N, NULL, xi, x, 0)) ;
        }
        Li = L->i ; Lx = L->x ; Ui = U->i ; Ux = U->x ;
        col = q ? (q [k]) : k ;
        top = cs_spsolve (L, A, col, xi, x, pinv, 1) ;  /* x = L\A(:,col) */
        /* --- Find pivot --------------------------------------------------- */
        ipiv = -1 ;
        a = -1 ;
        for (p = top ; p < n ; p++)
        {
            i = xi [p] ;            /* x(i) is nonzero */
            if (pinv [i] < 0)       /* row i is not yet pivotal */
            {
                if ((t = fabs (x [i])) > a)
                {
                    a = t ;         /* largest pivot candidate so far */
                    ipiv = i ;
                }
            }
            else                    /* x(i) is the entry U(pinv[i],k) */
            {
                Ui [unz] = pinv [i] ;
                Ux [unz++] = x [i] ;
            }
        }

        //if (ipiv == -1 ) printf ( "Pivot Failure\n" );
        fflush ( stdout );

        if (ipiv == -1 || a <= 0) return (cs_ndone (N, NULL, xi, x, 0)) ;
        if (pinv [col] < 0 && fabs (x [col]) >= a*tol) ipiv = col ;
        /* --- Divide by pivot ---------------------------------------------- */
        pivot = x [ipiv] ;          /* the chosen pivot */
        Ui [unz] = k ;              /* last entry in U(:,k) is U(k,k) */
        Ux [unz++] = pivot ;
        pinv [ipiv] = k ;           /* ipiv is the kth pivot row */
        Li [lnz] = ipiv ;           /* first entry in L(:,k) is L(k,k) = 1 */
        Lx [lnz++] = 1 ;
        for (p = top ; p < n ; p++) /* L(k+1:n,k) = x / pivot */
        {
            i = xi [p] ;
            if (pinv [i] < 0)       /* x(i) is an entry in L(:,k) */
            {
                Li [lnz] = i ;      /* save unpermuted row in L */
                Lx [lnz++] = x [i] / pivot ;    /* scale pivot column */
            }
            x [i] = 0 ;             /* x [0..n-1] = 0 for next k */
        }
    }
    /* --- Finalize L and U ------------------------------------------------- */
    Lp [n] = lnz ;
    Up [n] = unz ;
    Li = L->i ;                     /* fix row indices of L for final pinv */
    for (p = 0 ; p < lnz ; p++) Li [p] = pinv [Li [p]] ;
    cs_sprealloc (L, 0) ;           /* remove extra space from L and U */
    cs_sprealloc (U, 0) ;
    return (cs_ndone (N, NULL, xi, x, 1)) ;     /* success */
}

int cs_lsolve(const cs *L, double *x) {

	int p, j, n, *Lp, *Li;
	double *Lx;
	if (!CS_CSC (L) || !x)
		return (0); /* check inputs */
	n = L->n;
	Lp = L->p;
	Li = L->i;
	Lx = L->x;
	for (j = 0; j < n; j++) {
		x[j] /= Lx[Lp[j]];
		for (p = Lp[j] + 1; p < Lp[j + 1]; p++) {
			x[Li[p]] -= Lx[p] * x[j];
		}
	}
	return (1);
}

int cs_usolve (const cs *U, double *x)
{
    int p, j, n, *Up, *Ui ;
    double *Ux ;
    if (!CS_CSC (U) || !x) return (0) ;                     /* check inputs */
    n = U->n ; Up = U->p ; Ui = U->i ; Ux = U->x ;
    for (j = n-1 ; j >= 0 ; j--)
    {
        x [j] /= Ux [Up [j+1]-1] ;
        for (p = Up [j] ; p < Up [j+1]-1 ; p++)
        {
            x [Ui [p]] -= Ux [p] * x [j] ;
        }
    }
    return (1) ;
}

int cs_ltsolve(const cs *L, double *x) {

	int p, j, n, *Lp, *Li;
	double *Lx;
	if (!CS_CSC (L) || !x)
		return (0); /* check inputs */
	n = L->n;
	Lp = L->p;
	Li = L->i;
	Lx = L->x;
	for (j = n - 1; j >= 0; j--) {
		for (p = Lp[j] + 1; p < Lp[j + 1]; p++) {
			x[j] -= Lx[p] * x[Li[p]];
		}
		x[j] /= Lx[Lp[j]];
	}
	return (1);
}

int cs_lusol (int order, const cs *A, double *b, double tol)
{
    double *x ;
    css *S ;
    csn *N ;
    int n, ok ;
    if (!CS_CSC (A) || !b) return (0) ;     /* check inputs */
    n = A->n ;
    S = cs_sqr (order, A, 0) ;              /* ordering and symbolic analysis */
    //if (S == NULL) printf(	"Empty S\n" );
    N = cs_lu (A, S, tol) ;                 /* numeric LU factorization */
    //if (N == NULL) printf(	"Empty N\n" );
    x = cs_malloc (n, sizeof (double)) ;    /* get workspace */
    //if (x == NULL) printf(	"Empty x\n" );
    ok = (S && N && x) ;
    if (ok)
    {
    	//printf ( "Proceed to LU Factorization\n" );
        cs_ipvec (N->pinv, b, x, n) ;       /* x = b(p) */
        cs_lsolve (N->L, x) ;               /* x = L\x */
        cs_usolve (N->U, x) ;               /* x = U\x */
        cs_ipvec (S->q, x, b, n) ;          /* b(q) = x */
    }
    cs_free (x) ;
    cs_sfree (S) ;
    cs_nfree (N) ;
    return (ok) ;
}

static int cs_vcount (const cs *A, css *S)
{
    int i, k, p, pa, n = A->n, m = A->m, *Ap = A->p, *Ai = A->i, *next, *head,
        *tail, *nque, *pinv, *leftmost, *w, *parent = S->parent ;
    S->pinv = pinv = cs_malloc (m+n, sizeof (int)) ;        /* allocate pinv, */
    S->leftmost = leftmost = cs_malloc (m, sizeof (int)) ;  /* and leftmost */
    w = cs_malloc (m+3*n, sizeof (int)) ;   /* get workspace */
    if (!pinv || !w || !leftmost)
    {
        cs_free (w) ;                       /* pinv and leftmost freed later */
        return (0) ;                        /* out of memory */
    }
    next = w ; head = w + m ; tail = w + m + n ; nque = w + m + 2*n ;
    for (k = 0 ; k < n ; k++) head [k] = -1 ;   /* queue k is empty */
    for (k = 0 ; k < n ; k++) tail [k] = -1 ;
    for (k = 0 ; k < n ; k++) nque [k] = 0 ;
    for (i = 0 ; i < m ; i++) leftmost [i] = -1 ;
    for (k = n-1 ; k >= 0 ; k--)
    {
        for (p = Ap [k] ; p < Ap [k+1] ; p++)
        {
            leftmost [Ai [p]] = k ;         /* leftmost[i] = min(find(A(i,:)))*/
        }
    }
    for (i = m-1 ; i >= 0 ; i--)            /* scan rows in reverse order */
    {
        pinv [i] = -1 ;                     /* row i is not yet ordered */
        k = leftmost [i] ;
        if (k == -1) continue ;             /* row i is empty */
        if (nque [k]++ == 0) tail [k] = i ; /* first row in queue k */
        next [i] = head [k] ;               /* put i at head of queue k */
        head [k] = i ;
    }
    S->lnz = 0 ;
    S->m2 = m ;
    for (k = 0 ; k < n ; k++)               /* find row permutation and nnz(V)*/
    {
        i = head [k] ;                      /* remove row i from queue k */
        S->lnz++ ;                          /* count V(k,k) as nonzero */
        if (i < 0) i = S->m2++ ;            /* add a fictitious row */
        pinv [i] = k ;                      /* associate row i with V(:,k) */
        if (--nque [k] <= 0) continue ;     /* skip if V(k+1:m,k) is empty */
        S->lnz += nque [k] ;                /* nque [k] is nnz (V(k+1:m,k)) */
        if ((pa = parent [k]) != -1)        /* move all rows to parent of k */
        {
            if (nque [pa] == 0) tail [pa] = tail [k] ;
            next [tail [k]] = head [pa] ;
            head [pa] = next [i] ;
            nque [pa] += nque [k] ;
        }
    }
    for (i = 0 ; i < m ; i++) if (pinv [i] < 0) pinv [i] = k++ ;
    cs_free (w) ;
    return (1) ;
}

css *cs_sqr (int order, const cs *A, int qr)
{
    int n, k, ok = 1, *post ;
    css *S ;
    if (!CS_CSC (A)) return (NULL) ;        /* check inputs */
    n = A->n ;
    S = cs_calloc (1, sizeof (css)) ;       /* allocate result S */
    if (!S) return (NULL) ;                 /* out of memory */
    S->q = cs_amd (order, A) ;              /* fill-reducing ordering */
    if (order && !S->q) return (cs_sfree (S)) ;
    if (qr)                                 /* QR symbolic analysis */
    {
        cs *C = order ? cs_permute (A, NULL, S->q, 0) : ((cs *) A) ;
        S->parent = cs_etree (C, 1) ;       /* etree of C'*C, where C=A(:,q) */
        post = cs_post (S->parent, n) ;
        S->cp = cs_counts (C, S->parent, post, 1) ;  /* col counts chol(C'*C) */
        cs_free (post) ;
        ok = C && S->parent && S->cp && cs_vcount (C, S) ;
        if (ok) for (S->unz = 0, k = 0 ; k < n ; k++) S->unz += S->cp [k] ;
        ok = ok && S->lnz >= 0 && S->unz >= 0 ;     /* csi overflow guard */
        if (order) cs_spfree (C) ;
    }
    else
    {
        S->unz = 4*(A->p [n]) + n ;         /* for LU factorization only, */
        S->lnz = S->unz ;                   /* guess nnz(L) and nnz(U) */
    }
    return (ok ? S : cs_sfree (S)) ;        /* return result S */
}

int *cs_etree(const cs *A, int ata) {

	int i, k, p, m, n, inext, *Ap, *Ai, *w, *parent, *ancestor, *prev;
	if (!CS_CSC (A))
		return (NULL); /* check inputs */
	m = A->m;
	n = A->n;
	Ap = A->p;
	Ai = A->i;
	parent = (int *) cs_malloc(n, sizeof(int)); /* allocate result */
	w = (int *) cs_malloc(n + (ata ? m : 0), sizeof(int)); /* get workspace */
	if (!w || !parent)
		return (cs_idone(parent, NULL, w, 0));
	ancestor = w;
	prev = w + n;
	if (ata)
		for (i = 0; i < m; i++)
			prev[i] = -1;
	for (k = 0; k < n; k++) {
		parent[k] = -1; /* node k has no parent yet */
		ancestor[k] = -1; /* nor does k have an ancestor */
		for (p = Ap[k]; p < Ap[k + 1]; p++) {
			i = ata ? (prev[Ai[p]]) : (Ai[p]);
			for (; i != -1 && i < k; i = inext) /* traverse from i to k */
			{
				inext = ancestor[i]; /* inext = ancestor of i */
				ancestor[i] = k; /* path compression */
				if (inext == -1)
					parent[i] = k; /* no anc., parent is k */
			}
			if (ata)
				prev[Ai[p]] = k;
		}
	}
	return (cs_idone(parent, NULL, w, 1));
}

int cs_reach (cs *G, const cs *B, int k, int *xi, const int *pinv)
{
    int p, n, top, *Bp, *Bi, *Gp ;
    if (!CS_CSC (G) || !CS_CSC (B) || !xi) return (-1) ;    /* check inputs */
    n = G->n ; Bp = B->p ; Bi = B->i ; Gp = G->p ;
    top = n ;
    for (p = Bp [k] ; p < Bp [k+1] ; p++)
    {
        if (!CS_MARKED (Gp, Bi [p]))    /* start a dfs at unmarked node i */
        {
            top = cs_dfs (Bi [p], G, top, xi, xi+n, pinv) ;
        }
    }
    for (p = top ; p < n ; p++) CS_MARK (Gp, xi [p]) ;  /* restore G */
    return (top) ;
}

int cs_ereach(const cs *A, int k, const int *parent, int *s, int *w) {

	int i, p, n, len, top, *Ap, *Ai;
	if (!CS_CSC (A) || !parent || !s || !w)
		return (-1); /* check inputs */
	top = n = A->n;
	Ap = A->p;
	Ai = A->i;
	CS_MARK (w, k); /* mark node k as visited */
	for (p = Ap[k]; p < Ap[k + 1]; p++) {
		i = Ai[p]; /* A(i,k) is nonzero */
		if (i > k)
			continue; /* only use upper triangular part of A */
		for (len = 0; !CS_MARKED (w,i); i = parent[i]) /* traverse up etree*/
		{
			s[len++] = i; /* L(k,i) is nonzero */
			CS_MARK (w, i); /* mark i as visited */
		}
		while (len > 0)
			s[--top] = s[--len]; /* push path onto stack */
	}
	for (p = top; p < n; p++)
		CS_MARK (w, s [p]); /* unmark all nodes */
	CS_MARK (w, k); /* unmark node k */
	return (top); /* s [top..n-1] contains pattern of L(k,:)*/
}

int cs_dfs (int j, cs *G, int top, int *xi, int *pstack, const int *pinv)
{
    int i, p, p2, done, jnew, head = 0, *Gp, *Gi ;
    if (!CS_CSC (G) || !xi || !pstack) return (-1) ;    /* check inputs */
    Gp = G->p ; Gi = G->i ;
    xi [0] = j ;                /* initialize the recursion stack */
    while (head >= 0)
    {
        j = xi [head] ;         /* get j from the top of the recursion stack */
        jnew = pinv ? (pinv [j]) : j ;
        if (!CS_MARKED (Gp, j))
        {
            CS_MARK (Gp, j) ;       /* mark node j as visited */
            pstack [head] = (jnew < 0) ? 0 : CS_UNFLIP (Gp [jnew]) ;
        }
        done = 1 ;                  /* node j done if no unvisited neighbors */
        p2 = (jnew < 0) ? 0 : CS_UNFLIP (Gp [jnew+1]) ;
        for (p = pstack [head] ; p < p2 ; p++)  /* examine all neighbors of j */
        {
            i = Gi [p] ;            /* consider neighbor node i */
            if (CS_MARKED (Gp, i)) continue ;   /* skip visited node i */
            pstack [head] = p ;     /* pause depth-first search of node j */
            xi [++head] = i ;       /* start dfs at node i */
            done = 0 ;              /* node j is not done */
            break ;                 /* break, to start dfs (i) */
        }
        if (done)               /* depth-first search at node j is done */
        {
            head-- ;            /* remove j from the recursion stack */
            xi [--top] = j ;    /* and place in the output stack */
        }
    }
    return (top) ;
}

int cs_tdfs(int j, int k, int *head, const int *next, int *post, int *stack) {

	int i, p, top = 0;
	if (!head || !next || !post || !stack)
		return (-1); /* check inputs */
	stack[0] = j; /* place j on the stack */
	while (top >= 0) /* while (stack is not empty) */
	{
		p = stack[top]; /* p = top of stack */
		i = head[p]; /* i = youngest child of p */
		if (i == -1) {
			top--; /* p has no unordered children left */
			post[k++] = p; /* node p is the kth postordered node */
		} else {
			head[p] = next[i]; /* remove i from children of p */
			stack[++top] = i; /* start dfs on child node i */
		}
	}
	return (k);
}

int *cs_post(const int *parent, int n) {

	int j, k = 0, *post, *w, *head, *next, *stack;
	if (!parent)
		return (NULL); /* check inputs */
	post = (int *) cs_malloc(n, sizeof(int)); /* allocate result */
	w = (int *) cs_malloc(3 * n, sizeof(int)); /* get workspace */
	if (!w || !post)
		return (cs_idone(post, NULL, w, 0));
	head = w;
	next = w + n;
	stack = w + 2 * n;
	for (j = 0; j < n; j++)
		head[j] = -1; /* empty linked lists */
	for (j = n - 1; j >= 0; j--) /* traverse nodes in reverse order*/
	{
		if (parent[j] == -1)
			continue; /* j is a root */
		next[j] = head[parent[j]]; /* add j to list of its parent */
		head[parent[j]] = j;
	}
	for (j = 0; j < n; j++) {
		if (parent[j] != -1)
			continue; /* skip j if it is not a root */
		k = cs_tdfs(j, k, head, next, post, stack);
	}
	return (cs_idone(post, NULL, w, 1)); /* success; free w, return post */
}

int cs_leaf(int i, int j, const int *first, int *maxfirst, int *prevleaf, int *ancestor, int *jleaf) {

	int q, s, sparent, jprev;
	if (!first || !maxfirst || !prevleaf || !ancestor || !jleaf)
		return (-1);
	*jleaf = 0;
	if (i <= j || first[j] <= maxfirst[i])
		return (-1); /* j not a leaf */
	maxfirst[i] = first[j]; /* update max first[j] seen so far */
	jprev = prevleaf[i]; /* jprev = previous leaf of ith subtree */
	prevleaf[i] = j;
	*jleaf = (jprev == -1) ? 1 : 2; /* j is first or subsequent leaf */
	if (*jleaf == 1)
		return (i); /* if 1st leaf, q = root of ith subtree */
	for (q = jprev; q != ancestor[q]; q = ancestor[q])
		;
	/* EMPTY */
	for (s = jprev; s != q; s = sparent) {
		sparent = ancestor[s]; /* path compression */
		ancestor[s] = q;
	}
	return (q); /* q = least common ancester (jprev,j) */
}

void init_ata(cs *AT, const int *post, int *w, int **head, int **next) {

	int i, k, p, m = AT->n, n = AT->m, *ATp = AT->p, *ATi = AT->i;
	*head = w + 4 * n, *next = w + 5 * n + 1;
	for (k = 0; k < n; k++)
		w[post[k]] = k; /* invert post */
	for (i = 0; i < m; i++) {
		for (k = n, p = ATp[i]; p < ATp[i + 1]; p++)
			k = CS_MIN (k, w [ATi[p]]);
		(*next)[i] = (*head)[k]; /* place row i in linked list k */
		(*head)[k] = i;
	}
}

int *cs_counts(const cs *A, const int *parent, const int *post, int ata) {

	int i, j, k, n, m, J, s, p, q, jleaf, *ATp, *ATi, *maxfirst, *prevleaf, *ancestor,
			*head = NULL, *next = NULL, *colcount, *w, *first, *delta;
	cs *AT;
	if (!CS_CSC (A) || !parent || !post)
		return (NULL); /* check inputs */
	m = A->m;
	n = A->n;
	s = 4 * n + (ata ? (n + m + 1) : 0);
	delta = colcount = (int *) cs_malloc(n, sizeof(int)); /* allocate result */
	w = (int *) cs_malloc(s, sizeof(int)); /* get workspace */
	AT = cs_transpose(A, 0); /* AT = A' */
	if (!AT || !colcount || !w)
		return (cs_idone(colcount, AT, w, 0));
	ancestor = w;
	maxfirst = w + n;
	prevleaf = w + 2 * n;
	first = w + 3 * n;
	for (k = 0; k < s; k++)
		w[k] = -1; /* clear workspace w [0..s-1] */
	for (k = 0; k < n; k++) /* find first [j] */
	{
		j = post[k];
		delta[j] = (first[j] == -1) ? 1 : 0; /* delta[j]=1 if j is a leaf */
		for (; j != -1 && first[j] == -1; j = parent[j])
			first[j] = k;
	}
	ATp = AT->p;
	ATi = AT->i;
	if (ata)
		init_ata(AT, post, w, &head, &next);
	for (i = 0; i < n; i++)
		ancestor[i] = i; /* each node in its own set */
	for (k = 0; k < n; k++) {
		j = post[k]; /* j is the kth node in postordered etree */
		if (parent[j] != -1)
			delta[parent[j]]--; /* j is not a root */
		for (J = HEAD (k,j); J != -1; J = NEXT (J)) /* J=j for LL'=A case */
		{
			for (p = ATp[J]; p < ATp[J + 1]; p++) {
				i = ATi[p];
				q = cs_leaf(i, j, first, maxfirst, prevleaf, ancestor, &jleaf);
				if (jleaf >= 1)
					delta[j]++; /* A(i,j) is in skeleton */
				if (jleaf == 2)
					delta[q]--; /* account for overlap in q */
			}
		}
		if (parent[j] != -1)
			ancestor[j] = parent[j];
	}
	for (j = 0; j < n; j++) /* sum up delta's of each child */
	{
		if (parent[j] != -1)
			colcount[parent[j]] += colcount[j];
	}
	return (cs_idone(colcount, AT, w, 1)); /* success: free workspace */
}

int cs_wclear(int mark, int lemax, int *w, int n) {

	int k;
	if (mark < 2 || (mark + lemax < 0)) {
		for (k = 0; k < n; k++)
			if (w[k] != 0)
				w[k] = 1;
		mark = 2;
	}
	return (mark); /* at this point, w [0..n-1] < mark holds */
}

int cs_diag(int i, int j, double aij, void *other) {

	return (i != j);
}

int *cs_amd(int order, const cs *A) {/* order 0:natural, 1:Chol, 2:LU, 3:QR */

	cs *C, *A2, *AT;
	int *Cp, *Ci, *last, *W, *len, *nv, *next, *P, *head, *elen, *degree, *w, *hhead, *ATp, *ATi,
			d, dk, dext, lemax = 0, e, elenk, eln, i, j, k, k1, k2, k3, jlast, ln, dense, nzmax,
			mindeg = 0, nvi, nvj, nvk, mark, wnvi, ok, cnz, nel = 0, p, p1, p2, p3, p4, pj, pk,
			pk1, pk2, pn, q, n, m, t;
	unsigned int h;
	/* --- Construct matrix C ----------------------------------------------- */
	if (!CS_CSC (A) || order <= 0 || order > 3)
		return (NULL); /* check */
	AT = cs_transpose(A, 0); /* compute A' */
	if (!AT)
		return (NULL);
	m = A->m;
	n = A->n;
	dense = CS_MAX (16, 10 * sqrt ((double) n)); /* find dense threshold */
	dense = CS_MIN (n-2, dense);
	if (order == 1 && n == m) {
		C = cs_add(A, AT, 0, 0); /* C = A+A' */
	} else if (order == 2) {
		ATp = AT->p; /* drop dense columns from AT */
		ATi = AT->i;
		for (p2 = 0, j = 0; j < m; j++) {
			p = ATp[j]; /* column j of AT starts here */
			ATp[j] = p2; /* new column j starts here */
			if (ATp[j + 1] - p > dense)
				continue; /* skip dense col j */
			for (; p < ATp[j + 1]; p++)
				ATi[p2++] = ATi[p];
		}
		ATp[m] = p2; /* finalize AT */
		A2 = cs_transpose(AT, 0); /* A2 = AT' */
		C = A2 ? cs_multiply(AT, A2) : NULL; /* C=A'*A with no dense rows */
		cs_spfree(A2);
	} else {
		C = cs_multiply(AT, A); /* C=A'*A */
	}
	cs_spfree(AT);
	if (!C)
		return (NULL);
	cs_fkeep(C, &cs_diag, NULL); /* drop diagonal entries */
	Cp = C->p;
	cnz = Cp[n];
	P = (int *) cs_malloc(n + 1, sizeof(int)); /* allocate result */
	W = (int *) cs_malloc(8 * (n + 1), sizeof(int)); /* get workspace */
	t = cnz + cnz / 5 + 2 * n; /* add elbow room to C */
	if (!P || !W || !cs_sprealloc(C, t))
		return (cs_idone(P, C, W, 0));
	len = W;
	nv = W + (n + 1);
	next = W + 2 * (n + 1);
	head = W + 3 * (n + 1);
	elen = W + 4 * (n + 1);
	degree = W + 5 * (n + 1);
	w = W + 6 * (n + 1);
	hhead = W + 7 * (n + 1);
	last = P; /* use P as workspace for last */
	/* --- Initialize quotient graph ---------------------------------------- */
	for (k = 0; k < n; k++)
		len[k] = Cp[k + 1] - Cp[k];
	len[n] = 0;
	nzmax = C->nzmax;
	Ci = C->i;
	for (i = 0; i <= n; i++) {
		head[i] = -1; /* degree list i is empty */
		last[i] = -1;
		next[i] = -1;
		hhead[i] = -1; /* hash list i is empty */
		nv[i] = 1; /* node i is just one node */
		w[i] = 1; /* node i is alive */
		elen[i] = 0; /* Ek of node i is empty */
		degree[i] = len[i]; /* degree of node i */
	}
	mark = cs_wclear(0, 0, w, n); /* clear w */
	elen[n] = -2; /* n is a dead element */
	Cp[n] = -1; /* n is a root of assembly tree */
	w[n] = 0; /* n is a dead element */
	/* --- Initialize degree lists ------------------------------------------ */
	for (i = 0; i < n; i++) {
		d = degree[i];
		if (d == 0) /* node i is empty */
		{
			elen[i] = -2; /* element i is dead */
			nel++;
			Cp[i] = -1; /* i is a root of assembly tree */
			w[i] = 0;
		} else if (d > dense) /* node i is dense */
		{
			nv[i] = 0; /* absorb i into element n */
			elen[i] = -1; /* node i is dead */
			nel++;
			Cp[i] = CS_FLIP (n);
			nv[n]++;
		} else {
			if (head[d] != -1)
				last[head[d]] = i;
			next[i] = head[d]; /* put node i in degree list d */
			head[d] = i;
		}
	}
	while (nel < n) /* while (selecting pivots) do */
	{
		/* --- Select node of minimum approximate degree -------------------- */
		for (k = -1; mindeg < n && (k = head[mindeg]) == -1; mindeg++)
			;
		if (next[k] != -1)
			last[next[k]] = -1;
		head[mindeg] = next[k]; /* remove k from degree list */
		elenk = elen[k]; /* elenk = |Ek| */
		nvk = nv[k]; /* # of nodes k represents */
		nel += nvk; /* nv[k] nodes of A eliminated */
		/* --- Garbage collection ------------------------------------------- */
		if (elenk > 0 && cnz + mindeg >= nzmax) {
			for (j = 0; j < n; j++) {
				if ((p = Cp[j]) >= 0) /* j is a live node or element */
				{
					Cp[j] = Ci[p]; /* save first entry of object */
					Ci[p] = CS_FLIP (j); /* first entry is now CS_FLIP(j) */
				}
			}
			for (q = 0, p = 0; p < cnz;) /* scan all of memory */
			{
				if ((j = CS_FLIP (Ci [p++])) >= 0) /* found object j */
				{
					Ci[q] = Cp[j]; /* restore first entry of object */
					Cp[j] = q++; /* new pointer to object j */
					for (k3 = 0; k3 < len[j] - 1; k3++)
						Ci[q++] = Ci[p++];
				}
			}
			cnz = q; /* Ci [cnz...nzmax-1] now free */
		}
		/* --- Construct new element ---------------------------------------- */
		dk = 0;
		nv[k] = -nvk; /* flag k as in Lk */
		p = Cp[k];
		pk1 = (elenk == 0) ? p : cnz; /* do in place if elen[k] == 0 */
		pk2 = pk1;
		for (k1 = 1; k1 <= elenk + 1; k1++) {
			if (k1 > elenk) {
				e = k; /* search the nodes in k */
				pj = p; /* list of nodes starts at Ci[pj]*/
				ln = len[k] - elenk; /* length of list of nodes in k */
			} else {
				e = Ci[p++]; /* search the nodes in e */
				pj = Cp[e];
				ln = len[e]; /* length of list of nodes in e */
			}
			for (k2 = 1; k2 <= ln; k2++) {
				i = Ci[pj++];
				if ((nvi = nv[i]) <= 0)
					continue; /* node i dead, or seen */
				dk += nvi; /* degree[Lk] += size of node i */
				nv[i] = -nvi; /* negate nv[i] to denote i in Lk*/
				Ci[pk2++] = i; /* place i in Lk */
				if (next[i] != -1)
					last[next[i]] = last[i];
				if (last[i] != -1) /* remove i from degree list */
				{
					next[last[i]] = next[i];
				} else {
					head[degree[i]] = next[i];
				}
			}
			if (e != k) {
				Cp[e] = CS_FLIP (k); /* absorb e into k */
				w[e] = 0; /* e is now a dead element */
			}
		}
		if (elenk != 0)
			cnz = pk2; /* Ci [cnz...nzmax] is free */
		degree[k] = dk; /* external degree of k - |Lk\i| */
		Cp[k] = pk1; /* element k is in Ci[pk1..pk2-1] */
		len[k] = pk2 - pk1;
		elen[k] = -2; /* k is now an element */
		/* --- Find set differences ----------------------------------------- */
		mark = cs_wclear(mark, lemax, w, n); /* clear w if necessary */
		for (pk = pk1; pk < pk2; pk++) /* scan 1: find |Le\Lk| */
		{
			i = Ci[pk];
			if ((eln = elen[i]) <= 0)
				continue;/* skip if elen[i] empty */
			nvi = -nv[i]; /* nv [i] was negated */
			wnvi = mark - nvi;
			for (p = Cp[i]; p <= Cp[i] + eln - 1; p++) /* scan Ei */
			{
				e = Ci[p];
				if (w[e] >= mark) {
					w[e] -= nvi; /* decrement |Le\Lk| */
				} else if (w[e] != 0) /* ensure e is a live element */
				{
					w[e] = degree[e] + wnvi; /* 1st time e seen in scan 1 */
				}
			}
		}
		/* --- Degree update ------------------------------------------------ */
		for (pk = pk1; pk < pk2; pk++) /* scan2: degree update */
		{
			i = Ci[pk]; /* consider node i in Lk */
			p1 = Cp[i];
			p2 = p1 + elen[i] - 1;
			pn = p1;
			for (h = 0, d = 0, p = p1; p <= p2; p++) /* scan Ei */
			{
				e = Ci[p];
				if (w[e] != 0) /* e is an unabsorbed element */
				{
					dext = w[e] - mark; /* dext = |Le\Lk| */
					if (dext > 0) {
						d += dext; /* sum up the set differences */
						Ci[pn++] = e; /* keep e in Ei */
						h += e; /* compute the hash of node i */
					} else {
						Cp[e] = CS_FLIP (k); /* aggressive absorb. e->k */
						w[e] = 0; /* e is a dead element */
					}
				}
			}
			elen[i] = pn - p1 + 1; /* elen[i] = |Ei| */
			p3 = pn;
			p4 = p1 + len[i];
			for (p = p2 + 1; p < p4; p++) /* prune edges in Ai */
			{
				j = Ci[p];
				if ((nvj = nv[j]) <= 0)
					continue; /* node j dead or in Lk */
				d += nvj; /* degree(i) += |j| */
				Ci[pn++] = j; /* place j in node list of i */
				h += j; /* compute hash for node i */
			}
			if (d == 0) /* check for mass elimination */
			{
				Cp[i] = CS_FLIP (k); /* absorb i into k */
				nvi = -nv[i];
				dk -= nvi; /* |Lk| -= |i| */
				nvk += nvi; /* |k| += nv[i] */
				nel += nvi;
				nv[i] = 0;
				elen[i] = -1; /* node i is dead */
			} else {
				degree[i] = CS_MIN (degree [i], d); /* update degree(i) */
				Ci[pn] = Ci[p3]; /* move first node to end */
				Ci[p3] = Ci[p1]; /* move 1st el. to end of Ei */
				Ci[p1] = k; /* add k as 1st element in of Ei */
				len[i] = pn - p1 + 1; /* new len of adj. list of node i */
				h %= n; /* finalize hash of i */
				next[i] = hhead[h]; /* place i in hash bucket */
				hhead[h] = i;
				last[i] = h; /* save hash of i in last[i] */
			}
		} /* scan2 is done */
		degree[k] = dk; /* finalize |Lk| */
		lemax = CS_MAX (lemax, dk);
		mark = cs_wclear(mark + lemax, lemax, w, n); /* clear w */
		/* --- Supernode detection ------------------------------------------ */
		for (pk = pk1; pk < pk2; pk++) {
			i = Ci[pk];
			if (nv[i] >= 0)
				continue; /* skip if i is dead */
			h = last[i]; /* scan hash bucket of node i */
			i = hhead[h];
			hhead[h] = -1; /* hash bucket will be empty */
			for (; i != -1 && next[i] != -1; i = next[i], mark++) {
				ln = len[i];
				eln = elen[i];
				for (p = Cp[i] + 1; p <= Cp[i] + ln - 1; p++)
					w[Ci[p]] = mark;
				jlast = i;
				for (j = next[i]; j != -1;) /* compare i with all j */
				{
					ok = (len[j] == ln) && (elen[j] == eln);
					for (p = Cp[j] + 1; ok && p <= Cp[j] + ln - 1; p++) {
						if (w[Ci[p]] != mark)
							ok = 0; /* compare i and j*/
					}
					if (ok) /* i and j are identical */
					{
						Cp[j] = CS_FLIP (i); /* absorb j into i */
						nv[i] += nv[j];
						nv[j] = 0;
						elen[j] = -1; /* node j is dead */
						j = next[j]; /* delete j from hash bucket */
						next[jlast] = j;
					} else {
						jlast = j; /* j and i are different */
						j = next[j];
					}
				}
			}
		}
		/* --- Finalize new element------------------------------------------ */
		for (p = pk1, pk = pk1; pk < pk2; pk++) /* finalize Lk */
		{
			i = Ci[pk];
			if ((nvi = -nv[i]) <= 0)
				continue;/* skip if i is dead */
			nv[i] = nvi; /* restore nv[i] */
			d = degree[i] + dk - nvi; /* compute external degree(i) */
			d = CS_MIN (d, n - nel - nvi);
			if (head[d] != -1)
				last[head[d]] = i;
			next[i] = head[d]; /* put i back in degree list */
			last[i] = -1;
			head[d] = i;
			mindeg = CS_MIN (mindeg, d); /* find new minimum degree */
			degree[i] = d;
			Ci[p++] = i; /* place i in Lk */
		}
		nv[k] = nvk; /* # nodes absorbed into k */
		if ((len[k] = p - pk1) == 0) /* length of adj list of element k*/
		{
			Cp[k] = -1; /* k is a root of the tree */
			w[k] = 0; /* k is now a dead element */
		}
		if (elenk != 0)
			cnz = p; /* free unused space in Lk */
	}
	/* --- Postordering ----------------------------------------------------- */
	for (i = 0; i < n; i++)
		Cp[i] = CS_FLIP (Cp [i]);/* fix assembly tree */
	for (j = 0; j <= n; j++)
		head[j] = -1;
	for (j = n; j >= 0; j--) /* place unordered nodes in lists */
	{
		if (nv[j] > 0)
			continue; /* skip if j is an element */
		next[j] = head[Cp[j]]; /* place j in list of its parent */
		head[Cp[j]] = j;
	}
	for (e = n; e >= 0; e--) /* place elements in lists */
	{
		if (nv[e] <= 0)
			continue; /* skip unless e is an element */
		if (Cp[e] != -1) {
			next[e] = head[Cp[e]]; /* place e in list of its parent */
			head[Cp[e]] = e;
		}
	}
	for (k = 0, i = 0; i <= n; i++) /* postorder the assembly tree */
	{
		if (Cp[i] == -1)
			k = cs_tdfs(i, k, head, next, P, w);
	}
	return (cs_idone(P, C, W, 1));
}

css *cs_schol(int order, const cs *A) {

	int n, *c, *post, *P;
	cs *C;
	css *S;
	if (!CS_CSC (A))
		return (NULL); /* check inputs */
	n = A->n;
	S = (css *) cs_calloc(1, sizeof(css)); /* allocate result S */
	if (!S)
		return (NULL); /* out of memory */
	P = cs_amd(order, A); /* P = amd(A+A'), or natural */
	S->pinv = cs_pinv(P, n); /* find inverse permutation */
	cs_free(P);
	if (order && !S->pinv)
		return (cs_sfree(S));
	C = cs_symperm(A, S->pinv, 0); /* C = spones(triu(A(P,P))) */
	S->parent = cs_etree(C, 0); /* find etree of C */
	post = cs_post(S->parent, n); /* postorder the etree */
	c = cs_counts(C, S->parent, post, 0); /* find column counts of chol(C) */
	cs_free(post);
	cs_spfree(C);
	S->cp = (int *) cs_malloc(n + 1, sizeof(int)); /* allocate result S->cp */
	S->unz = S->lnz = cs_cumsum(S->cp, c, n); /* find column pointers for L */
	cs_free(c);
	return ((S->lnz >= 0) ? S : cs_sfree(S));
}

csn *cs_chol(const cs *A, const css *S) {

	double d, lki;
	double *Lx, *x, *Cx;
	int top, i, p, k, n, *Li, *Lp, *cp, *pinv, *s, *c, *parent, *Cp, *Ci;
	cs *L, *C, *E;
	csn *N;
	if (!CS_CSC (A) || !S || !S->cp || !S->parent)
		return (NULL);
	n = A->n;
	N = (csn *) cs_calloc(1, sizeof(csn)); /* allocate result */
	c = (int *) cs_malloc(2 * n, sizeof(int)); /* get int workspace */
	x = (double *) cs_malloc(n, sizeof(double)); /* get double workspace */
	cp = S->cp;
	pinv = S->pinv;
	parent = S->parent;
	C = pinv ? cs_symperm(A, pinv, 1) : ((cs *) A);
	E = pinv ? C : NULL; /* E is alias for A, or a copy E=A(p,p) */
	if (!N || !c || !x || !C)
		return (cs_ndone(N, E, c, x, 0));
	s = c + n;
	Cp = C->p;
	Ci = C->i;
	Cx = C->x;
	N->L = L = cs_spalloc(n, n, cp[n], 1, 0); /* allocate result */
	if (!L)
		return (cs_ndone(N, E, c, x, 0));
	Lp = L->p;
	Li = L->i;
	Lx = L->x;
	for (k = 0; k < n; k++)
		Lp[k] = c[k] = cp[k];
	for (k = 0; k < n; k++) /* compute L(k,:) for L*L' = C */
	{
		/* --- Nonzero pattern of L(k,:) ------------------------------------ */
		top = cs_ereach(C, k, parent, s, c); /* find pattern of L(k,:) */
		x[k] = 0; /* x (0:k) is now zero */
		for (p = Cp[k]; p < Cp[k + 1]; p++) /* x = full(triu(C(:,k))) */
		{
			if (Ci[p] <= k)
				x[Ci[p]] = Cx[p];
		}
		d = x[k]; /* d = C(k,k) */
		x[k] = 0; /* clear x for k+1st iteration */
		/* --- Triangular solve --------------------------------------------- */
		for (; top < n; top++) /* solve L(0:k-1,0:k-1) * x = C(:,k) */
		{
			i = s[top]; /* s [top..n-1] is pattern of L(k,:) */
			lki = x[i] / Lx[Lp[i]]; /* L(k,i) = x (i) / L(i,i) */
			x[i] = 0; /* clear x for k+1st iteration */
			for (p = Lp[i] + 1; p < c[i]; p++) {
				x[Li[p]] -= Lx[p] * lki;
			}
			d -= lki * lki; /* d = d - L(k,i)*L(k,i) */
			p = c[i]++;
			Li[p] = k; /* store L(k,i) in column i */
			Lx[p] = lki;
		}
		/* --- Compute L(k,k) ----------------------------------------------- */
		if (d <= 0)
			return (cs_ndone(N, E, c, x, 0)); /* not pos def */
		p = c[k]++;
		Li[p] = k; /* store L(k,k) = sqrt (d) in column k */
		Lx[p] = sqrt(d);
	}
	Lp[n] = cp[n]; /* finalize L */
	return (cs_ndone(N, E, c, x, 1)); /* success: free E,s,x; return N */
}

int cs_rechol(const cs *A, const csn *N, int *pinv, int *c, double *x) {

	double d, lki;
	double *Lx, *Cx;
	int t, i, p, k, n, *Li, *Lp, *Ui, *Up, *Cp, *Ci;
	cs *C, *E;
	if (!CS_CSC (A) || !N || !N->L || !N->U || !c || !x)
		return (0);
	n = A->n;
	C = pinv ? cs_symperm(A, pinv, 1) : ((cs *) A);
	E = pinv ? C : NULL; /* E is alias for A, or a copy E=A(p,p) */
	if (!C) {
		cs_spfree(E);
		return (0);
	}
	Cp = C->p;
	Ci = C->i;
	Cx = C->x;
	Lp = N->L->p;
	Li = N->L->i;
	Lx = N->L->x;
	Up = N->U->p;
	Ui = N->U->i;
	for (k = 0; k < n; k++) /* compute L(k,:) for L*L' = C */
	{
		x[k] = 0; /* x (0:k) is now zero */
		for (p = Cp[k]; p < Cp[k + 1]; p++) /* x = full(triu(C(:,k))) */
		{
			if (Ci[p] <= k)
				x[Ci[p]] = Cx[p];
		}
		d = x[k]; /* d = C(k,k) */
		x[k] = 0; /* clear x for k+1st iteration */
		for (t = Up[k]; t < Up[k + 1] - 1; t++) /* solve L(0:k-1,0:k-1) * x = C(:,k) */
		/* the loop does not include the diagonal element of row k of L - or column k of U - */
		/* (which is placed at the end of column k for a sorted matrix - after its transposition), */
		/* because it is treated outside the loop */
		{
			i = Ui[t]; /* pattern of L(k,:) */
			lki = x[i] / Lx[Lp[i]]; /* L(k,i) = x (i) / L(i,i) */
			x[i] = 0; /* clear x for k+1st iteration */
			for (p = Lp[i] + 1; p < c[i]; p++) {
				x[Li[p]] -= Lx[p] * lki;
			}
			d -= lki * lki; /* d = d - L(k,i)*L(k,i) */
			p = c[i]++;
			Lx[p] = lki;
		}
		if (d <= 0) {
			cs_spfree(E);
			return (0);
		} /* not pos def */
		p = c[k]++;
		Lx[p] = sqrt(d);
	}
	cs_spfree(E);
	return (1); /* success */
}

int cs_cholsol (int order, const cs *A, double *b)
{
    double *x ;
    css *S ;
    csn *N ;
    int n, ok ;
    if (!CS_CSC (A) || !b) return (0) ;     /* check inputs */
    n = A->n ;
    S = cs_schol (order, A) ;               /* ordering and symbolic analysis */
    N = cs_chol (A, S) ;                    /* numeric Cholesky factorization */
    x = cs_malloc (n, sizeof (double)) ;    /* get workspace */
    ok = (S && N && x) ;
    if (ok)
    {
        cs_ipvec (S->pinv, b, x, n) ;   /* x = P*b */
        cs_lsolve (N->L, x) ;           /* x = L\x */
        cs_ltsolve (N->L, x) ;          /* x = L'\x */
        cs_pvec (S->pinv, x, b, n) ;    /* b = P'*x */
    }
    cs_free (x) ;
    cs_sfree (S) ;
    cs_nfree (N) ;
    return (ok) ;
}

int cs_reltol(cs *A, double tol) {

	int j, p, q, nz = 0, n, *Ap, *Ai;
	double *Ax;
	if (!CS_CSC (A))
		return (-1); /* check inputs */
	n = A->n;
	Ap = A->p;
	Ai = A->i;
	Ax = A->x;
	for (j = 0; j < n; j++) {
		p = q = Ap[j]; /* get current location of col j */
		Ap[j] = nz; /* record new location of col j */
		for (; p < Ap[j + 1]; p++) {
			if (fabs(Ax ? Ax[p] / Ax[q] : 1) > tol) {
				if (Ax)
					Ax[nz] = Ax[p]; /* keep A(i,j) */
				Ai[nz++] = Ai[p];
			}
		}
	}
	Ap[n] = nz; /* finalize A */
	cs_sprealloc(A, 0); /* remove extra space from A */
	return (nz);
}

int cs_spsolve (cs *G, const cs *B, int k, int *xi, double *x, const int *pinv, int lo)
{
    int j, J, p, q, px, top, n, *Gp, *Gi, *Bp, *Bi ;
    double *Gx, *Bx ;
    if (!CS_CSC (G) || !CS_CSC (B) || !xi || !x) return (-1) ;
    Gp = G->p ; Gi = G->i ; Gx = G->x ; n = G->n ;
    Bp = B->p ; Bi = B->i ; Bx = B->x ;
    top = cs_reach (G, B, k, xi, pinv) ;        /* xi[top..n-1]=Reach(B(:,k)) */
    for (p = top ; p < n ; p++) x [xi [p]] = 0 ;    /* clear x */
    for (p = Bp [k] ; p < Bp [k+1] ; p++) x [Bi [p]] = Bx [p] ; /* scatter B */
    for (px = top ; px < n ; px++)
    {
        j = xi [px] ;                               /* x(j) is nonzero */
        J = pinv ? (pinv [j]) : j ;                 /* j maps to col J of G */
        if (J < 0) continue ;                       /* column J is empty */
        x [j] /= Gx [lo ? (Gp [J]) : (Gp [J+1]-1)] ;/* x(j) /= G(j,j) */
        p = lo ? (Gp [J]+1) : (Gp [J]) ;            /* lo: L(j,j) 1st entry */
        q = lo ? (Gp [J+1]) : (Gp [J+1]-1) ;        /* up: U(j,j) last entry */
        for ( ; p < q ; p++)
        {
            x [Gi [p]] -= Gx [p] * x [j] ;          /* x(i) -= G(i,j) * x(j) */
        }
    }
    return (top) ;                                  /* return top of stack */
}
/********************************************************************************
 *                                                                              *
 *                            UTILITY FUNCTIONS                                 *
 *                                                                              *
 ********************************************************************************/

int cs_entry(cs *T, int i, int j, double x) {

	if (!CS_TRIPLET (T) || i < 0 || j < 0)
		return (0); /* check inputs */
	if (T->nz >= T->nzmax && !cs_sprealloc(T, 2 * (T->nzmax)))
		return (0);
	if (T->x)
		T->x[T->nz] = x;
	T->i[T->nz] = i;
	T->p[T->nz++] = j;
	T->m = CS_MAX (T->m, i+1);
	T->n = CS_MAX (T->n, j+1);
	return (1);
}

cs *cs_load(char *matrixFilename) {

	double i, j; /* use double for integers to avoid csi conflicts */
	double x;
	cs *T;
	FILE *matrixFilePtr;

	matrixFilePtr = fopen(matrixFilename, "r");
	if (matrixFilePtr == NULL) {

		fprintf(stderr, "Could not open output file %s for writing\n", matrixFilename);
		exit(EXIT_FAILURE);
	}

	T = cs_spalloc(0, 0, 1, 1, 1); /* allocate result */
	while (fscanf(matrixFilePtr, "%lg %lg %lg\n", &i, &j, &x) == 3) {
		if (!cs_entry(T, i, j, x))
			return (cs_spfree(T));
	}
	return (T);
}

int cs_print(const cs *A, const char *outputFilename, int brief) {

	int p, j, m, n, nzmax, nz, *Ap, *Ai;
	double *Ax;
	FILE *outputFilePtr;

	outputFilePtr = fopen(outputFilename, "w");
	if (outputFilePtr == NULL) {

		fprintf(stderr, "Could not open output file %s for writing\n", outputFilename);
		exit(EXIT_FAILURE);
	}

	if (!A) {
		fprintf(outputFilePtr, "(null)\n");
		return (0);
	}
	m = A->m;
	n = A->n;
	Ap = A->p;
	Ai = A->i;
	Ax = A->x;
	nzmax = A->nzmax;
	nz = A->nz;
	if (nz < 0) {
		fprintf(outputFilePtr, "%d-by-%d, nzmax: %d nnz: %d\n", m, n, nzmax, Ap[n]);
		for (j = 0; j < n; j++) {
			fprintf(outputFilePtr, "    col %d : locations %d to %d\n", j, Ap[j], Ap[j + 1] - 1);
			for (p = Ap[j]; p < Ap[j + 1]; p++) {
				fprintf(outputFilePtr, "      %d : %g\n", Ai[p], Ax ? Ax[p] : 1);
				if (brief && p > 20) {
					fprintf(outputFilePtr, "  ...\n");
					return (1);
				}
			}
		}
	} else {
		fprintf(outputFilePtr, "triplet: %d-by-%d, nzmax: %d nnz: %d\n", m, n, nzmax, nz);
		for (p = 0; p < nz; p++) {
			fprintf(outputFilePtr, "    %d %d : %g\n", Ai[p], Ap[p], Ax ? Ax[p] : 1);
			if (brief && p > 20) {
				fprintf(outputFilePtr, "  ...\n");
				return (1);
			}
		}
	}

	fclose(outputFilePtr);
	return (1);
}
