"""
Dixon works for
        number of polynomials m == number of variables n + 1
        
        else 
            raise ValueError('Method invalid for given combination.')

"""


from sympy import IndexedBase, Matrix, Mul, Poly, var, pprint
from sympy import rem, prod, degree_list
from sympy.core.compatibility import range
from sympy.polys.monomials import monomial_deg
from sympy.polys.monomials import itermonomials
from sympy.polys.orderings import monomial_key
from sympy.polys.polytools import poly_from_expr
from sympy.functions.combinatorial.factorials import binomial

from itertools import combinations_with_replacement

from sympy import symbols, solve, factor, expand, simplify, nsimplify

from operator import itemgetter

from sympy.polys.subresultants_qq_zz import *

class DixonResultant():
    """
    A class for retrieving the Dixon's resultant of a multivariate
    system.

    Examples
    ========
    >>> from sympy.core import symbols

    >>> from sympy.polys.multivariate_resultants import DixonResultant
    >>> x, y = symbols('x, y')

    >>> p = x + y
    >>> q = x ** 2 + y ** 3
    >>> h = x ** 2 + y

    >>> dixon = DixonResultant(variables=[x, y], polynomials=[p, q, h])
    >>> poly = dixon.get_dixon_polynomial()
    >>> matrix = dixon.get_dixon_matrix(polynomial=poly)
    >>> matrix
    Matrix([
    [ 0,  0, -1,  0, -1],
    [ 0, -1,  0, -1,  0],
    [-1,  0,  1,  0,  0],
    [ 0, -1,  0,  0,  1],
    [-1,  0,  0,  1,  0]])
    >>> matrix.det()
    0

    Reference
    ==========
    1. [Kapur1994]_
    2. [Palancz08]_

    See Also
    ========
    Notebook in examples: sympy/example/notebooks.
    """

    def __init__(self, polynomials, variables):
        """
        A class that takes two lists, a list of polynomials and list of
        variables. Returns the Dixon matrix of the multivariate system.

        Parameters
        ----------
        polynomials : list of polynomials
            A  list of m n-degree polynomials
        variables: list
            A list of all n variables
        """
        self.polynomials = polynomials
        self.variables = variables

        self.n = len(self.variables)
        self.m = len(self.polynomials)

        a = IndexedBase("alpha")
        # A list of n alpha variables (the replacing variables)
        self.dummy_variables = [a[i] for i in range(self.n)]
        
        # List of all_monomials, to compute the max degree of each variable
        # all_monomials = []
        # for j in range(len(self.polynomials)):
        #    all_monomials.extend(Poly(self.polynomials[j], *self.variables).monoms())

        
        # A list of the d_max of each variable.   
        self.max_degrees = [max([degree_list(poly) for poly in self.polynomials],
                key=itemgetter(i))[i] for i in range(len(self.variables))]
        
        #print('all_monomials = ', all_monomials)
        #print('\n')        
        print('max-degrees = ', self.max_degrees)
        print('\n')

    def get_dixon_polynomial(self):
        r"""
        Returns
        -------

        dixon_polynomial: polynomial
            Dixon's polynomial is calculated as:

            delta = Delta(A) / ((x_1 - a_1) ... (x_n - a_n)) where,

            A =  |p_1(x_1,... x_n), ..., p_n(x_1,... x_n)|
                 |p_1(a_1,... x_n), ..., p_n(a_1,... x_n)|
                 |...             , ...,              ...|
                 |p_1(a_1,... a_n), ..., p_n(a_1,... a_n)|
        """
        if self.m != (self.n + 1):
            raise ValueError('Method invalid for given combination.')

        # First row
        rows = [self.polynomials]

        temp = list(self.variables)

        for idx in range(self.n):
            temp[idx] = self.dummy_variables[idx]
            substitution = {var: t for var, t in zip(self.variables, temp)}
            rows.append([f.subs(substitution) for f in self.polynomials])

        A = Matrix(rows)

        terms = zip(self.variables, self.dummy_variables)
        product_of_differences = Mul(*[a - b for a, b in terms])
        dixon_polynomial = (A.det() / product_of_differences).factor()

        return poly_from_expr(dixon_polynomial, self.dummy_variables)[0]


    def get_upper_degree(self):
        # AGA 02-02-2019 :: There was originally the factor (i + 1) in the 
        # exponent below but was removed as not needed.
        # Also replaced (self.max_degrees[i] - 1) by self.max_degrees[i]
        list_of_products = [self.variables[i] ** self.max_degrees[i]
                            for i in range(self.n)]
        ## see change in __init__ in this regard
        print('list_of_products =', list_of_products)
        print('\n')
        product = prod(list_of_products)
        print('product =', product)
        print('\n')
        product = Poly(product).monoms()
        print('product =', product)
        print('\n')
        print('monomial_deg(*product) = ', monomial_deg(*product))
        print('\n')

        return monomial_deg(*product)

    def get_dixon_matrix(self, polynomial):
        r"""
        Construct the Dixon matrix from the coefficients of polynomial
        \alpha. Each coefficient is viewed as a polynomial of x_1, ...,
        x_n.
        """

        # A list of coefficients (in x_i, ..., x_n terms) of the power
        # products a_1, ..., a_n in Dixon's polynomial.
        coefficients = polynomial.coeffs()
        print('coefficients = ', coefficients)
        print('\n')
        monomials = list(itermonomials(self.variables,
                                       self.get_upper_degree()))
        monomials = sorted(monomials, reverse=True,
                           key=monomial_key('lex', self.variables))
        print('sorted monomials = ', monomials)
        print('\n')

        dixon_matrix = Matrix([[Poly(c, *self.variables).coeff_monomial(m)
                                for m in monomials]
                                for c in coefficients])
        # pprint(dixon_matrix)
        print('dixon_matrix.shape = ', dixon_matrix.shape)
        print('\n')

        keep = [column for column in range(dixon_matrix.shape[-1])
                if any([element != 0 for element
                        in dixon_matrix[:, column]])]
        print('keep = ', keep)
        print('\n')
        return dixon_matrix[:, keep]


a, b, c, d, x, y, z = symbols('a b c d x y z')

f = x**3 - 2*x**2 - 11*x + 12 #  (x-1)(x+3)(x-4)
g = x**2 + 3*x - 4            #  (x+4)(x-1)

print('f = ', f, '\n')
print('g = ', g, '\n')


from sympy import solve_poly_system  # Groebner bases
print('solve_poly_system([f, g], x) = ', solve_poly_system([f, g], x), '\n')

from sympy import groebner
print(groebner([f, g], x, order='lex'), '\n', '\n')

print('f = ', f.as_poly(), '\n')


f = x**3 - 2*x**2 - 11*x + 12.2
print('f = ', f, '\n')
print('f = ', f.as_poly(), '\n')
print('nsimplify(f) = ', nsimplify(f).as_poly(), '\n')


f = x**3 - 2*x**2 - 11*x + 12/5
print('f = ', f, '\n')
print('f = ', f.as_poly(), '\n')



"""
# Gena's Example
print('Gena Example')

f = z ** 3 * y * x - z * x ** 2
g = z ** 2 * y ** 2 - x
h = z * y ** 2 * x - z * y * x

print('f = ', f, '\n')
print('g = ', g, '\n')
print('h = ', h, '\n')

dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det(), '\n')



f = x**3 - 7*x + 7
g = 3*x ** 2 - 7 


dixon = DixonResultant(variables=[x], polynomials=[f, g])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly)
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')


# Example 1  (Common root exists but cannot be found since det = 0)
print('Example 1 --- Common root exists but cannot be found since det = 0', '\n')

f = x**3 - 2*x**2 - 11*x + 12 #  (x-1)(x+3)(x-4)
g = x**2 + 3*x - 4            #  (x+4)(x-1)
print('f = ', f)
print('g = ', g)

print('\n')


dixon = DixonResultant(variables=[x], polynomials=[f, g])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det(), '\n')

# Example 2  (NO common roots)
print('Example 2 --- NO common roots', '\n')

f = x**3 - 2*x**2 - 11*x + 12  # (x-1)(x+3)(x-4)
g = x**2 + 5*x + 4             # (x+4)(x+1)
print('f = ', f)
print('g = ', g)

print('\n')


dixon = DixonResultant(variables=[x], polynomials=[g, f])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det(), '\n')



# Example 3  (Parametric coefficients; Full root recovery; det != 0)
print('Example 3 --- Parametric coefficients; Full root recovery; det != 0', '\n')

A = symbols('A')

f = x**2*A + 3*x**2 - x*A - A**3 - 2*A**2 - 3*x + 3*A # (x+A-1)(A+3)(x-A)
g = x**2 + 4*x*A + 3*A**2                             # (x+3A)(x+A)
print('f = ', f)
print('g = ', g)

print('\n')


dixon = DixonResultant(variables=[x], polynomials=[f, g])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
if matrix.shape[0] == matrix.shape[1]:
    print('det = ',matrix.det().factor(), '\n')
else:
    raise ValueError('Matrix non-square.')

# Setting det = 0 and solving for A yields (without multiplicities):
print('Setting det = 0 and solving for A yields (without multiplicities):')
print('\n')
print(solve(matrix.det().factor(), A))
print('\n')

# The true solution of f(x,A)=0, g(x,A)=0 over QQ is:
print('The true solution of f(x,A)=0, g(x,A)=0 over QQ is:', '\n')
from sympy import solve_poly_system  # Groebner bases
print(solve_poly_system([f, g], x, A))
print('\n')

# The above solution can be found from det = 0 as follows:
print('The above solution can be found from det = 0 as follows:', '\n')
for value in solve(matrix.det().factor(), A):
    print('for A = ',  value, 'the roots of f(x,A)=0, g(x,A)=0 over QQ are :')
    print('f = ', solve(f.subs(A, value), x))
    print('g = ', solve(g.subs(A, value), x))
    print('\n')
print('\n')


# Example 4  (Parametric coefficients; NO root information; det = 0)
print('Example 4 --- Parametric coefficients; NO root information; det = 0', '\n')

A = symbols('A')

f = (expand((A + 3)*(x - 1)*(x - A)))
g = x**2 + x*A - x - A
print('f = ', f)
print('g = ', g)

print('\n')


dixon = DixonResultant(variables=[x], polynomials=[f, g])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')


# In the following examples we illustrate Dixon's method and
# its limitations on polynomials that are not necessarily
# generic or ndegree



print("In the following examples we illustrate Dixon's method and")
print("its limitations on polynomials that are not necessarily")
print("generic or ndegree")
print('\n')

# EXAMPLE 5  Det !=0, so, we expect no solutions (which is true)
print('EXAMPLE 5 --- Det !=0, so, we expect no solutions (which is true)','\n')

b, y = symbols('b y')

# f = x**2*A - x**2*A**2 + 3*x**2 - 4*x*A + A**2 - 3*x + 3*A
f = expand((x - 3)*(y + 3))
g = expand((x + 3)*(y - 3))
h = expand((x - 2)*(y + 2))
print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')


dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
print('coefficients = ', poly.coeffs(), '\n')

matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')


# Example 6; Det = 0 ==> No information
print('Example 6; Det = 0 ==> No information')
print('\n')
x, y = symbols('x y')
f = expand((x - y)*(y + 3))
g = expand((x + 3*y)*(x + y))
h = expand((x - 2*y)*(x + y))
print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')


dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')


# Example 7;

# Here Det != 0 but system has trivial solution; that is, 
# the vanishing of the determinant of the new Dixon matrix is not even a 
# necessary condition for the existence of solutions of the system
print('Example 7 ')
print('\n')
print('Here, Det != 0 but system has trivial solution; that is, ')
print('the vanishing of the determinant of the Dixon matrix is not even a ')
print('necessary condition for the existence of solutions of the system')
print('\n')
x, y = symbols('x y')
f = expand((x - 3)*y)
g = expand(x*(y - 3))
h = expand((x - 2)*y)
print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')


dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')


# Example 8; Det = 0, hence No root information
print('Example 8; Det = 0, hence No root information')
print('\n')

A = symbols('A')
x, y = symbols('x y')

f = x*y + x*A + x -A**2 - A + y**2 + y  # y + A + 1 - A + x + y 
g = x**2 + x*A - x + x*y + y*A - y      # x + y*x - 1 + A      
h = x**2 + x*y + 2*x - x*A - y*A - 2*A  # x + y + 2*x - A

dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')



# Example 9; matrix is NOT square, hence NO det, hence NO root information
print('Example 9; matrix is NOT square, hence NO det, hence NO root information')
print('Example from isaac 1994')
print('\n')

a, b, c, x, y = symbols('a b c x y')

f = a*x**2 + b*x*y + (b + c - a)*x + a*y + 3*(c - 1)
g = 2*a**2*x**2 + 2*a*b*x*y +a*b*y + b**2      
h = 4*(a - b)*x + c*(a + b)*y + 4*a*b
print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')


dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly.as_expr(), '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
print(matrix)
print('\n')
print('matrix dimensions = ',matrix.shape)
print('\n')
print('matrix[0, 0] = ', matrix[0,0])
print('\n')


# Example 10; matrix is square
print('Example 10; matrix is square')
print('Example from kapur 1999')
print('\n')

a, b, c, x, y = symbols('a b c x y')

f = x**2 + a*x*y - y + b
g = x*y + (a + b)*y - c     
h = b*x + y + 2*a*b
print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')


dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
print(matrix)
print('\n')
pprint(matrix)
print('\n')
print('matrix dimensions = ',matrix.shape)
print('\n')
print('det = ', matrix.det().factor())
print('\n')


# Example 11, by Palancz et all p. 507 (2008); Full root recovery; det != 0
print('Example 11, by Palancz et all p. 507 (2008); Full root recovery; det != 0')
print('\n')

π = symbols('π')

f = 12 - 11*x - 2*x**2 + x**3 # = (x - 1)*(x + 3)*(x - 4) 
g = -4*π + 4*x - π*x + x**2   # = (x - π)*(x + π)
print('f = ', f)
print('g = ', g)

print('\n')

polynomials = [f, g]
variables = [x]

dixon = DixonResultant(variables=[x], polynomials=[g, f])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')
print('\n')

# The true solution of f(x)=0, g(x, π)=0 over QQ is:
print('The true solution of f(x)=0, g(x, π)=0  over QQ is:', '\n')
from sympy import solve_poly_system
print(solve_poly_system([f, g], *[x, π]))
print('\n')

# The above solution can be found from det = 0 as follows:
print('The above solution can be found from det = 0 as follows:', '\n')

print('the roots of det= 0 are: ', solve(matrix.det().factor(), π))
print('\n')
for value in solve(matrix.det().factor(), π):
    print('for π = ',  value, 'the roots of f(x)=0, g(x, π)=0 over QQ are :')
    print('g = ', solve(g.subs(π, value), x))
    print('f = ', f.subs({x:value}))
    print('\n')


# Example 12, by Palancz et all (2008); Full root recovery; det != 0
print('Example 12, by Palancz et all (2008); Full root recovery; det != 0')
print('\n')

z =symbols('z')

f = x**2 + y**2 - 1
g = x**2 + z**2 - 1
h = y**2 + z**2 -1

print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')

polynomials = [f, g, h]
variables = [x,y]

dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')
print('\n')

# The true solution of f(x,y,z)=0, g(x,y,z)=0, h(x,y,z)=0 over QQ is:
print('The true solution of f(x,y,z)=0, g(x,y,z)=0, h(x,y,z)=0 over QQ is:', '\n')
from sympy import solve_poly_system
print(solve_poly_system([f, g, h], *[x,y,z]))
print('\n')

# The above solution can be found from det = 0 as follows:
print('The above solution can be found from det = 0 as follows:', '\n')

print('the roots of det= 0 are: ', solve((2*z**2 - 1)**4, z))
print('\n')
for value in solve(matrix.det().factor(), z):
    print('for z = ',  value, 'the roots of f(x,y,z)=0, g(x,y,z)=0, h(x,y,z)=0 over QQ are :')
    print('h = ', solve(h.subs(z, value), y))
    print('g = ', solve(g.subs(z, value), x))
    for valueY in solve(h.subs(z, value), y):
        print('y = ',valueY)
        for valueX in solve(g.subs(z, value), x):
            print('x = ',valueX)
            print('f = ', f.subs({x:valueX, y:valueY}))
    print('\n')
    

# Example 13 from Kapur 99
print('Example 13 from Kapur 99')

f = x**2 + a*x*y - y + b
g = x*y + (a + b)*y - c
h = b*x + y + 2*a*c

print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')

polynomials = [f, g, h]
variables = [x,y]

dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')
print('\n')
pprint(backward_eye(3)*matrix*backward_eye(3))
print('\n')


######################
# Geometric reasoning using Dixon resultants
# Kapur 99
######################
print('######################')
print('Geometric reasoning using Dixon resultants')
print('Kapur 99')
print('######################')
# Example 14 (Side-bisector relation) from Kapur 99, p. 104
print('Example 14 (Side-bisector relation) from Kapur 99, p. 104')

a_i, a_e, b_e = symbols('a_i a_e b_e')

f = a_i**2*(b + c)**2 - c*b*(c + b - a)*(c + b + a) 
g = a_e**2*(c - b)**2 - c*b*(a + b - c)*(c - b + a)
h = b_e**2*(c - a)**2 - a*c*(a + b - c)*(c + b - a)
print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')


# Eliminate b and c from the above

polynomials = [f, g, h]
variables = [a, b, c]

dixon = DixonResultant(variables=[b, c], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
# pprint(matrix)
print('\n')
print('dimensions = ',matrix.shape, '\n')
print('\n')

print('######################')
print('######################')
print('\n')


# Example 15 (Stiler p. 10)
print('Example 15 (Stiler p. 10)')

x, y = symbols('x y')

f = y + 3*x + 5
g = x**2 + y**2 - 5
h = y - x**3 + 3*x**2 - 3*x + 1

print('f = ', f)
print('g = ', g)
print('h = ', h)
print('\n')

dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')


print(solve([f, g, h],[x, y]), '\n')

from sympy import groebner
print(groebner([f, g, h ], x, y, order='lex'), '\n')


######################
# Below we rework Examples 3, 4, 8, and 9 and compute
# the Kapur-Saxena-Yang Dixon resultant
# as done in Nakos' paper

#
# NEED AN ALGORITHM TO ROW REDUCE WITHOUT SCALING
#
######################
print('######################')
print('Below we REWORK Examples 3, 4, 8, and 9 and compute')
print('the Kapur-Saxena-Yang Dixon resultant')
print('as done in Nakos paper')
print('\n')
print('NEED AN ALGORITHM TO ROW REDUCE WITHOUT SCALING')
print('######################')
print('\n')

# Example 3  REWORKED === ROW REDUCED RESULTANT; Full root recovery; det != 0
print('Example 3 REWORKED === ROW REDUCED RESULTANT; Full root recovery; det != 0', '\n')

A = symbols('A')

f = x**2*A + 3*x**2 - x*A - A**3 - 2*A**2 - 3*x + 3*A # (x+A-1)(A+3)(x-A)
g = x**2 + 4*x*A + 3*A**2                             # (x+3A)(x+A)

dixon = DixonResultant(variables=[x], polynomials=[f, g])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('Since the first column of the Dixon matrix is not a scalar multiple ')
print('of the second, the precondition of the Theorem applies')
print('\n')
print('ROW REDUCED MATRIX = ', '\n')
L, U, _ = matrix.LUdecomposition()
pprint(U)
# pprint(matrix.rref())
print('\n')
# polyA, the product of the diagonal elements, is the Dixon resultant
print('polyA, the product of the diagonal elements, is the Dixon resultant')
print('\n')
polyA = factor(simplify(U[0, 0] * U[1, 1]))
print('polyA = ', polyA)
print('\n')


# Setting polyA = 0 and solving for A yields (without multiplicities):
print('Setting polyA = 0 and solving for A yields (without multiplicities):')
print('\n')
print(solve(polyA, A))
print('\n')

# The true solution of f(x,A)=0, g(x,A)=0 over QQ is:
print('The true solution of f(x,A)=0, g(x,A)=0 over QQ is:', '\n')
from sympy import solve_poly_system
print(solve_poly_system([f, g], x, A))
print('\n')

# The above solution can be found from polyA = 0 as follows:
print('The above solution can be found from polyA = 0 as follows:', '\n')
for value in solve(polyA, A):
    print('for A = ',  value, 'the roots of f(x,A)=0, g(x,A)=0 over QQ are :')
    print('f = ', solve(f.subs(A, value), x))
    print('g = ', solve(g.subs(A, value), x))
    print('\n')
print('\n')


# Example 4  REWORKED === ROW REDUCED RESULTANT; PARTIAL root recovery; det = 0
print('Example 4 REWORKED === ROW REDUCED RESULTANT; PARTIAL root recovery; det = 0', '\n')

A = symbols('A')

f = (expand((A + 3)*(x - 1)*(x - A)))
g = x**2 + x*A - x - A

dixon = DixonResultant(variables=[x], polynomials=[f, g])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')
print('\n')
print('Since the first column of the Dixon matrix is  a scalar multiple ')
print('of the second, the precondition of the Theorem does NOT apply')
print('Reducing anyway we obtain:')
print('\n')
print('ROW REDUCED MATRIX = ', '\n')
L, U, _ = matrix.LUdecomposition()
pprint(U)
print('\n')
# polyA, the product of the diagonal elements, is the Dixon resultant
print('polyA, the product of the diagonal elements, is the Dixon resultant')
print('\n')
polyA = factor(simplify(U[0, 0] * U[1, 1]))
print('polyA = ', polyA)
print('\n')
pprint(matrix.rref())
print('\n')
polyA = U[1]
print('polyA = ', polyA)

# The true solution of f(x,A)=0, g(x,A)=0 over QQ is:
print('The true solution of f(x,A)=0, g(x,A)=0 over QQ is:', '\n')
from sympy import solve_poly_system
print(solve_poly_system([f, g], x))
print('\n')


# Example 8; REWORKED === ROW REDUCED RESULTANT; FULL root recovery; det = 0
print('Example 8; REWORKED === ROW REDUCED RESULTANT; FULL root recovery; det = 0')
print('\n')

A = symbols('A')

f = x*y + x*A + x -A**2 - A + y**2 + y  # y + A + 1 - A + x + y 
g = x**2 + x*A - x + x*y + y*A - y      # x + y*x - 1 + A      
h = x**2 + x*y + 2*x - x*A - y*A - 2*A  # x + y + 2*x - A

dixon = DixonResultant(variables=[x, y], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
pprint(matrix)
print('\n')
print('det = ',matrix.det().factor(), '\n')

print('First we show that the precondition of the Theorem ')
print('holds by proving that the LAST(?) column of the')
print('Dixon matrix M is not a linear combination of the remaining ones')
print('\n')
pprint(matrix.subs({A:1}).rref())
print('\n')
L, U, _ = matrix.LUdecomposition()
# pprint(U)
pprint(matrix.rref())
print('\n')
# polyA, the product of the diagonal elements, is the Dixon resultant
print('polyA, the product of the diagonal elements, is the Dixon resultant')
print('\n')
polyA = factor(simplify(U[0, 0] * U[1, 1]))
print('polyA = ', polyA)
print('\n')


print('#######', '\n')

# Heymann's problem (see Saxena's Ph.D. Thesis)
print('Heymann problem (see Saxena Ph.D. Thesis)', '\n')
print('#######', '\n')

a, b, c, ai, ae, be = symbols('a, b, c, ai, ae, be')


#######

f = ( (ai)**2 * (b + c)**2 - c*b*(c + b - a)*(c + b + a) ).subs({ai:3, ae:5, be:2})

g = ( (ae)**2 * (c - b)**2 - c*b*(a + b - c)*(c - b + a) ).subs({ai:3, ae:5, be:2})

h = ( (be)**2 * (c - a)**2 - a*c*(a + b - c)*(c + b - a) ).subs({ai:3, ae:5, be:2})
                          
print('f = ',f, '\n')
print('g = ',g, '\n')
print('h = ',h, '\n')

dixon = DixonResultant(variables=[b, c], polynomials=[f, g, h])
poly = dixon.get_dixon_polynomial()
print('poly = ', poly, '\n')
matrix = dixon.get_dixon_matrix(polynomial=poly)
print('matrix shape = ', matrix.shape, '\n')
# pprint(matrix)
print('\n')
if matrix.shape[0] == matrix.shape[1]:
    print('det = ',matrix.det().factor(), '\n')
else:
    raise ValueError('Matrix non-square.')


from sympy import groebner
print(groebner([f, g, h ], b, c, order='lex'), '\n')
"""